/*
 * Decompiled with CFR 0.152.
 */
package jetbrains.exodus.util;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Calendar;
import java.util.zip.GZIPOutputStream;
import jetbrains.exodus.ExodusException;
import jetbrains.exodus.backup.BackupStrategy;
import jetbrains.exodus.backup.Backupable;
import jetbrains.exodus.util.ByteArraySpinAllocator;
import jetbrains.exodus.util.IOUtil;
import org.apache.commons.compress.archivers.ArchiveEntry;
import org.apache.commons.compress.archivers.ArchiveOutputStream;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveOutputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipArchiveOutputStream;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CompressBackupUtil {
    private static final Logger logger = LoggerFactory.getLogger(CompressBackupUtil.class);

    private CompressBackupUtil() {
    }

    @NotNull
    public static File backup(@NotNull Backupable target, @NotNull File backupRoot, @Nullable String backupNamePrefix, boolean zip) throws Exception {
        if (!backupRoot.exists() && !backupRoot.mkdirs()) {
            throw new IOException("Failed to create " + backupRoot.getAbsolutePath());
        }
        String fileName = zip ? CompressBackupUtil.getTimeStampedZipFileName() : CompressBackupUtil.getTimeStampedTarGzFileName();
        File backupFile = new File(backupRoot, backupNamePrefix == null ? fileName : backupNamePrefix + fileName);
        return CompressBackupUtil.backup(target, backupFile, zip);
    }

    @NotNull
    public static File backup(@NotNull Backupable target, @NotNull File backupFile, boolean zip) throws Exception {
        if (backupFile.exists()) {
            throw new IOException("Backup file already exists:" + backupFile.getAbsolutePath());
        }
        BackupStrategy strategy = target.getBackupStrategy();
        strategy.beforeBackup();
        try {
            TarArchiveOutputStream archive;
            if (zip) {
                ZipArchiveOutputStream zipArchive = new ZipArchiveOutputStream((OutputStream)new BufferedOutputStream(new FileOutputStream(backupFile)));
                zipArchive.setLevel(9);
                archive = zipArchive;
            } else {
                archive = new TarArchiveOutputStream((OutputStream)new GZIPOutputStream(new BufferedOutputStream(new FileOutputStream(backupFile))));
            }
            try (TarArchiveOutputStream aos = archive;){
                for (BackupStrategy.FileDescriptor fd : strategy.listFiles()) {
                    long fileSize;
                    if (strategy.isInterrupted()) {
                        break;
                    }
                    File file = fd.getFile();
                    if (!file.isFile() || (fileSize = Math.min(fd.getFileSize(), strategy.acceptFile(file))) <= 0L) continue;
                    CompressBackupUtil.archiveFile((ArchiveOutputStream)aos, fd.getPath(), file, fileSize);
                }
            }
            if (strategy.isInterrupted()) {
                logger.info("Backup interrupted, deleting \"" + backupFile.getName() + "\"...");
                IOUtil.deleteFile((File)backupFile);
            } else {
                logger.info("Backup file \"" + backupFile.getName() + "\" created.");
            }
        }
        catch (Throwable t) {
            strategy.onError(t);
            throw ExodusException.toExodusException((Throwable)t, (String)"Backup failed");
        }
        finally {
            strategy.afterBackup();
        }
        return backupFile;
    }

    @NotNull
    public static String getTimeStampedTarGzFileName() {
        StringBuilder builder = new StringBuilder(30);
        CompressBackupUtil.appendTimeStamp(builder);
        builder.append(".tar.gz");
        return builder.toString();
    }

    @NotNull
    public static String getTimeStampedZipFileName() {
        StringBuilder builder = new StringBuilder(30);
        CompressBackupUtil.appendTimeStamp(builder);
        builder.append(".zip");
        return builder.toString();
    }

    public static void tar(@NotNull File source, @NotNull File dest) throws IOException {
        if (!source.exists()) {
            throw new IllegalArgumentException("No source file or folder exists: " + source.getAbsolutePath());
        }
        if (dest.exists()) {
            throw new IllegalArgumentException("Destination refers to existing file or folder: " + dest.getAbsolutePath());
        }
        TarArchiveOutputStream tarOut = null;
        try {
            tarOut = new TarArchiveOutputStream((OutputStream)new GZIPOutputStream((OutputStream)new BufferedOutputStream(new FileOutputStream(dest)), 4096));
            CompressBackupUtil.doTar("", source, tarOut);
            tarOut.close();
        }
        catch (IOException e) {
            CompressBackupUtil.cleanUp(tarOut, dest);
            throw e;
        }
    }

    private static void cleanUp(TarArchiveOutputStream tarOut, File dest) {
        if (tarOut != null) {
            try {
                tarOut.close();
            }
            catch (IOException iOException) {
                // empty catch block
            }
        }
        IOUtil.deleteFile((File)dest);
    }

    private static void doTar(String pathInArchive, File source, TarArchiveOutputStream tarOut) throws IOException {
        if (source.isDirectory()) {
            for (File file : IOUtil.listFiles((File)source)) {
                CompressBackupUtil.doTar(pathInArchive + source.getName() + File.separator, file, tarOut);
            }
        } else {
            CompressBackupUtil.archiveFile((ArchiveOutputStream)tarOut, pathInArchive, source, source.length());
        }
    }

    public static void archiveFile(@NotNull ArchiveOutputStream out, @NotNull String pathInArchive, @NotNull File source, long fileSize) throws IOException {
        TarArchiveEntry entry;
        if (!source.isFile()) {
            throw new IllegalArgumentException("Provided source is not a file: " + source.getAbsolutePath());
        }
        if (out instanceof TarArchiveOutputStream) {
            entry = new TarArchiveEntry(pathInArchive + source.getName());
            entry.setSize(fileSize);
            entry.setModTime(source.lastModified());
            out.putArchiveEntry((ArchiveEntry)entry);
        } else if (out instanceof ZipArchiveOutputStream) {
            entry = new ZipArchiveEntry(pathInArchive + source.getName());
            entry.setSize(fileSize);
            entry.setTime(source.lastModified());
            out.putArchiveEntry((ArchiveEntry)entry);
        } else {
            throw new IOException("Unknown archive output stream");
        }
        try (FileInputStream input = new FileInputStream(source);){
            IOUtil.copyStreams((InputStream)input, (long)fileSize, (OutputStream)out, (ByteArraySpinAllocator)IOUtil.BUFFER_ALLOCATOR);
        }
        out.closeArchiveEntry();
    }

    private static void appendTimeStamp(StringBuilder builder) {
        Calendar now = Calendar.getInstance();
        builder.append(now.get(1));
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(2) + 1);
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(5));
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(11));
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(12));
        builder.append('-');
        CompressBackupUtil.appendZeroPadded(builder, now.get(13));
    }

    private static void appendZeroPadded(StringBuilder builder, int value) {
        if (value < 10) {
            builder.append('0');
        }
        builder.append(value);
    }
}

