/*
 * Decompiled with CFR 0.152.
 */
package jetbrains.exodus.env;

import java.util.Map;
import java.util.NavigableMap;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import jetbrains.exodus.ExodusException;
import jetbrains.exodus.core.dataStructures.hash.HashMap;
import jetbrains.exodus.env.Environment;
import jetbrains.exodus.env.TransactionAcquireTimeoutException;
import jetbrains.exodus.env.TransactionBase;
import org.jetbrains.annotations.NotNull;

final class ReentrantTransactionDispatcher {
    private final int availablePermits;
    @NotNull
    private final Map<Thread, Integer> threadPermits;
    @NotNull
    private final NavigableMap<Long, Condition> regularQueue;
    @NotNull
    private final NavigableMap<Long, Condition> nestedQueue;
    @NotNull
    private final ReentrantLock lock;
    private long acquireOrder;
    private int acquiredPermits;

    ReentrantTransactionDispatcher(int maxSimultaneousTransactions) {
        if (maxSimultaneousTransactions < 1) {
            throw new IllegalArgumentException("maxSimultaneousTransactions < 1");
        }
        this.availablePermits = maxSimultaneousTransactions;
        this.threadPermits = new HashMap();
        this.regularQueue = new TreeMap<Long, Condition>();
        this.nestedQueue = new TreeMap<Long, Condition>();
        this.lock = new ReentrantLock(false);
        this.acquireOrder = 0L;
        this.acquiredPermits = 0;
    }

    int getAvailablePermits() {
        try (CriticalSection ignored = new CriticalSection(this.lock);){
            int n = this.availablePermits - this.acquiredPermits;
            return n;
        }
    }

    int acquireTransaction(@NotNull Thread thread) {
        try (CriticalSection ignored = new CriticalSection(this.lock);){
            int currentThreadPermits = this.getThreadPermitsToAcquire(thread);
            this.waitForPermits(thread, currentThreadPermits > 0 ? this.nestedQueue : this.regularQueue, 1, currentThreadPermits);
        }
        return 1;
    }

    int acquireExclusiveTransaction(@NotNull Thread thread) {
        try (CriticalSection ignored = new CriticalSection(this.lock);){
            int currentThreadPermits = this.getThreadPermitsToAcquire(thread);
            if (currentThreadPermits == 0) {
                this.waitForPermits(thread, this.regularQueue, this.availablePermits, 0);
                int n = this.availablePermits;
                return n;
            }
            this.waitForPermits(thread, this.nestedQueue, 1, currentThreadPermits);
        }
        return 1;
    }

    void acquireTransaction(@NotNull TransactionBase txn, @NotNull Environment env) {
        int acquiredPermits;
        Thread creatingThread = txn.getCreatingThread();
        if (txn.isExclusive()) {
            if (txn.isGCTransaction()) {
                int gcTransactionAcquireTimeout = env.getEnvironmentConfig().getGcTransactionAcquireTimeout();
                acquiredPermits = this.tryAcquireExclusiveTransaction(creatingThread, gcTransactionAcquireTimeout);
                if (acquiredPermits == 0) {
                    throw new TransactionAcquireTimeoutException(gcTransactionAcquireTimeout);
                }
            } else {
                acquiredPermits = this.acquireExclusiveTransaction(creatingThread);
            }
            if (acquiredPermits == 1) {
                txn.setExclusive(false);
            }
        } else {
            acquiredPermits = this.acquireTransaction(creatingThread);
        }
        txn.setAcquiredPermits(acquiredPermits);
    }

    void releaseTransaction(@NotNull Thread thread, int permits) {
        try (CriticalSection ignored = new CriticalSection(this.lock);){
            int currentThreadPermits = this.getThreadPermits(thread);
            if (permits > currentThreadPermits) {
                throw new ExodusException("Can't release more permits than it was acquired");
            }
            this.acquiredPermits -= permits;
            if ((currentThreadPermits -= permits) == 0) {
                this.threadPermits.remove(thread);
            } else {
                this.threadPermits.put(thread, currentThreadPermits);
            }
            this.notifyNextWaiters();
        }
    }

    void releaseTransaction(@NotNull TransactionBase txn) {
        this.releaseTransaction(txn.getCreatingThread(), txn.getAcquiredPermits());
    }

    void downgradeTransaction(@NotNull Thread thread, int permits) {
        if (permits > 1) {
            try (CriticalSection ignored = new CriticalSection(this.lock);){
                int currentThreadPermits = this.getThreadPermits(thread);
                if (permits > currentThreadPermits) {
                    throw new ExodusException("Can't release more permits than it was acquired");
                }
                this.acquiredPermits -= permits - 1;
                this.threadPermits.put(thread, currentThreadPermits -= permits - 1);
                this.notifyNextWaiters();
            }
        }
    }

    void downgradeTransaction(@NotNull TransactionBase txn) {
        this.downgradeTransaction(txn.getCreatingThread(), txn.getAcquiredPermits());
        txn.setAcquiredPermits(1);
    }

    int getThreadPermits(@NotNull Thread thread) {
        Integer result = this.threadPermits.get(thread);
        return result == null ? 0 : result;
    }

    private void waitForPermits(@NotNull Thread thread, @NotNull NavigableMap<Long, Condition> queue, int permits, int currentThreadPermits) {
        Condition condition = this.lock.newCondition();
        long currentOrder = this.acquireOrder++;
        queue.put(currentOrder, condition);
        while (this.acquiredPermits > this.availablePermits - permits || (Long)queue.firstKey() != currentOrder) {
            condition.awaitUninterruptibly();
        }
        queue.pollFirstEntry();
        this.acquiredPermits += permits;
        this.threadPermits.put(thread, currentThreadPermits + permits);
        if (this.acquiredPermits < this.availablePermits) {
            this.notifyNextWaiters();
        }
    }

    private int tryAcquireExclusiveTransaction(@NotNull Thread thread, int timeout) {
        long nanos = TimeUnit.MILLISECONDS.toNanos(timeout);
        try (CriticalSection ignored = new CriticalSection(this.lock);){
            if (this.getThreadPermits(thread) > 0) {
                throw new ExodusException("Exclusive transaction can't be nested");
            }
            Condition condition = this.lock.newCondition();
            long currentOrder = this.acquireOrder++;
            this.regularQueue.put(currentOrder, condition);
            while (this.acquiredPermits > 0 || (Long)this.regularQueue.firstKey() != currentOrder) {
                try {
                    if ((nanos = condition.awaitNanos(nanos)) >= 0L) continue;
                    break;
                }
                catch (InterruptedException e) {
                    // empty catch block
                    break;
                }
            }
            if (this.acquiredPermits == 0 && (Long)this.regularQueue.firstKey() == currentOrder) {
                this.regularQueue.pollFirstEntry();
                this.acquiredPermits = this.availablePermits;
                this.threadPermits.put(thread, this.availablePermits);
                int n = this.availablePermits;
                return n;
            }
            this.regularQueue.remove(currentOrder);
            this.notifyNextWaiters();
        }
        return 0;
    }

    private void notifyNextWaiters() {
        if (!ReentrantTransactionDispatcher.notifyNextWaiter(this.nestedQueue)) {
            ReentrantTransactionDispatcher.notifyNextWaiter(this.regularQueue);
        }
    }

    private int getThreadPermitsToAcquire(@NotNull Thread thread) {
        int currentThreadPermits = this.getThreadPermits(thread);
        if (currentThreadPermits == this.availablePermits) {
            throw new ExodusException("No more permits are available to acquire a transaction");
        }
        return currentThreadPermits;
    }

    private static boolean notifyNextWaiter(@NotNull NavigableMap<Long, Condition> queue) {
        if (!queue.isEmpty()) {
            queue.firstEntry().getValue().signal();
            return true;
        }
        return false;
    }

    private static final class CriticalSection
    implements AutoCloseable {
        @NotNull
        private final ReentrantLock lock;

        private CriticalSection(@NotNull ReentrantLock lock) {
            this.lock = lock;
            lock.lock();
        }

        @Override
        public void close() {
            this.lock.unlock();
        }
    }
}

