/*
 * Decompiled with CFR 0.152.
 */
package jetbrains.exodus.gc;

import java.io.File;
import java.lang.ref.SoftReference;
import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedQueue;
import jetbrains.exodus.ExodusException;
import jetbrains.exodus.core.dataStructures.LongArrayList;
import jetbrains.exodus.core.dataStructures.Priority;
import jetbrains.exodus.core.dataStructures.hash.IntHashMap;
import jetbrains.exodus.core.dataStructures.hash.LongHashSet;
import jetbrains.exodus.core.dataStructures.hash.LongSet;
import jetbrains.exodus.core.execution.Job;
import jetbrains.exodus.core.execution.JobProcessorAdapter;
import jetbrains.exodus.env.EnvironmentConfig;
import jetbrains.exodus.env.EnvironmentImpl;
import jetbrains.exodus.env.StoreImpl;
import jetbrains.exodus.env.TransactionAcquireTimeoutException;
import jetbrains.exodus.env.TransactionImpl;
import jetbrains.exodus.gc.BackgroundCleaner;
import jetbrains.exodus.gc.UtilizationProfile;
import jetbrains.exodus.io.RemoveBlockType;
import jetbrains.exodus.log.ExpiredLoggableInfo;
import jetbrains.exodus.log.Log;
import jetbrains.exodus.log.LogUtil;
import jetbrains.exodus.log.LoggableIterator;
import jetbrains.exodus.log.NewFileListener;
import jetbrains.exodus.log.RandomAccessLoggable;
import jetbrains.exodus.util.DeferredIO;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class GarbageCollector {
    public static final String UTILIZATION_PROFILE_STORE_NAME = "exodus.gc.up";
    private static final Logger logger = LoggerFactory.getLogger(GarbageCollector.class);
    @NotNull
    private final EnvironmentImpl env;
    @NotNull
    private final EnvironmentConfig ec;
    @NotNull
    private final UtilizationProfile utilizationProfile;
    @NotNull
    private final LongSet pendingFilesToDelete;
    @NotNull
    private final ConcurrentLinkedQueue<Long> deletionQueue;
    @NotNull
    private final BackgroundCleaner cleaner;
    private volatile int newFiles;
    @NotNull
    private final IntHashMap<StoreImpl> openStoresCache;
    private boolean useRegularTxn;

    public GarbageCollector(@NotNull EnvironmentImpl env) {
        this.env = env;
        this.ec = env.getEnvironmentConfig();
        this.pendingFilesToDelete = new LongHashSet();
        this.deletionQueue = new ConcurrentLinkedQueue();
        this.utilizationProfile = new UtilizationProfile(env, this);
        this.cleaner = new BackgroundCleaner(this);
        this.newFiles = this.ec.getGcFilesInterval() + 1;
        this.openStoresCache = new IntHashMap();
        env.getLog().addNewFileListener(new NewFileListener(){

            @Override
            public void fileCreated(long fileAddress) {
                ++GarbageCollector.this.newFiles;
                GarbageCollector.this.utilizationProfile.estimateTotalBytes();
                if (!GarbageCollector.this.cleaner.isCleaning() && GarbageCollector.this.newFiles > GarbageCollector.this.ec.getGcFilesInterval() && GarbageCollector.this.isTooMuchFreeSpace()) {
                    GarbageCollector.this.wake();
                }
            }
        });
    }

    public void clear() {
        this.utilizationProfile.clear();
        this.pendingFilesToDelete.clear();
        this.deletionQueue.clear();
        this.openStoresCache.clear();
        this.resetNewFiles();
    }

    public void setCleanerJobProcessor(final @NotNull JobProcessorAdapter processor) {
        this.cleaner.getJobProcessor().queue(new Job(){

            protected void execute() throws Throwable {
                GarbageCollector.this.cleaner.setJobProcessor(processor);
            }
        }, Priority.highest);
    }

    public void wake() {
        if (this.ec.isGcEnabled()) {
            this.env.executeTransactionSafeTask(new Runnable(){

                @Override
                public void run() {
                    GarbageCollector.this.cleaner.queueCleaningJob();
                }
            });
        }
    }

    void wakeAt(long millis) {
        if (this.ec.isGcEnabled()) {
            this.cleaner.queueCleaningJobAt(millis);
        }
    }

    int getMaximumFreeSpacePercent() {
        return 100 - this.ec.getGcMinUtilization();
    }

    public void fetchExpiredLoggables(@NotNull Iterable<ExpiredLoggableInfo> loggables) {
        this.utilizationProfile.fetchExpiredLoggables(loggables);
    }

    public long getFileFreeBytes(long fileAddress) {
        return this.utilizationProfile.getFileFreeBytes(fileAddress);
    }

    public void suspend() {
        this.cleaner.suspend();
    }

    public void resume() {
        this.cleaner.resume();
    }

    public void finish() {
        this.cleaner.finish();
    }

    @NotNull
    public UtilizationProfile getUtilizationProfile() {
        return this.utilizationProfile;
    }

    boolean isTooMuchFreeSpace() {
        return this.utilizationProfile.totalFreeSpacePercent() > this.getMaximumFreeSpacePercent();
    }

    public boolean doCleanFile(long fileAddress) {
        return this.doCleanFiles(Collections.singleton(fileAddress).iterator());
    }

    public static boolean isUtilizationProfile(@NotNull String storeName) {
        return UTILIZATION_PROFILE_STORE_NAME.equals(storeName);
    }

    @NotNull
    BackgroundCleaner getCleaner() {
        return this.cleaner;
    }

    int getMinFileAge() {
        return this.ec.getGcFileMinAge();
    }

    void deletePendingFiles() {
        Long fileAddress;
        this.cleaner.checkThread();
        LongArrayList filesToDelete = new LongArrayList();
        while ((fileAddress = this.deletionQueue.poll()) != null) {
            if (!this.pendingFilesToDelete.remove((Object)fileAddress)) continue;
            filesToDelete.add(fileAddress.longValue());
        }
        if (!filesToDelete.isEmpty()) {
            this.env.flushAndSync();
            for (long file : filesToDelete.toArray()) {
                this.removeFile(file);
            }
        }
    }

    @NotNull
    EnvironmentImpl getEnvironment() {
        return this.env;
    }

    Log getLog() {
        return this.env.getLog();
    }

    boolean cleanFiles(@NotNull Iterator<Long> fragmentedFiles) {
        this.cleaner.checkThread();
        return this.doCleanFiles(fragmentedFiles);
    }

    boolean isFileCleaned(long file) {
        return this.pendingFilesToDelete.contains(file);
    }

    void resetNewFiles() {
        this.newFiles = 0;
    }

    void setUseRegularTxn(boolean useRegularTxn) {
        this.useRegularTxn = useRegularTxn;
    }

    void cleanWholeLog() {
        this.cleaner.cleanWholeLog();
    }

    void testDeletePendingFiles() {
        long[] files = this.pendingFilesToDelete.toLongArray();
        boolean aFileWasDeleted = false;
        for (long file : files) {
            this.utilizationProfile.removeFile(file);
            this.getLog().removeFile(file, this.ec.getGcRenameFiles() ? RemoveBlockType.Rename : RemoveBlockType.Delete);
            aFileWasDeleted = true;
        }
        if (aFileWasDeleted) {
            this.pendingFilesToDelete.clear();
            this.utilizationProfile.estimateTotalBytes();
        }
    }

    static void loggingInfo(@NotNull String message) {
        if (logger.isInfoEnabled()) {
            logger.info(message);
        }
    }

    private boolean doCleanFiles(@NotNull Iterator<Long> fragmentedFiles) {
        TransactionImpl txn;
        if (!fragmentedFiles.hasNext()) {
            return true;
        }
        LongHashSet cleanedFiles = new LongHashSet();
        try {
            txn = this.useRegularTxn ? (TransactionImpl)this.env.beginTransaction() : this.env.beginGCTransaction();
        }
        catch (TransactionAcquireTimeoutException ignore) {
            return false;
        }
        boolean isTxnExclusive = txn.isExclusive();
        try {
            SoftReference<Object> oomeGuard = new SoftReference<Object>(new Object());
            long started = System.currentTimeMillis();
            while (fragmentedFiles.hasNext()) {
                long fileAddress = fragmentedFiles.next();
                this.cleanSingleFile(fileAddress, txn);
                cleanedFiles.add(fileAddress);
                if (isTxnExclusive && started + (long)this.ec.getGcTransactionTimeout() >= System.currentTimeMillis() && oomeGuard.get() != null) continue;
                break;
            }
            if (!txn.forceFlush()) {
                if (isTxnExclusive) {
                    throw new ExodusException("Can't be: exclusive txn should be successfully flushed");
                }
                boolean bl = false;
                return bl;
            }
        }
        catch (Throwable e) {
            throw ExodusException.toExodusException((Throwable)e);
        }
        finally {
            txn.abort();
        }
        if (!cleanedFiles.isEmpty()) {
            for (Long file : cleanedFiles) {
                this.pendingFilesToDelete.add((Object)file);
                this.utilizationProfile.removeFile(file);
            }
            this.utilizationProfile.estimateTotalBytes();
            this.env.executeTransactionSafeTask(new Runnable((LongSet)cleanedFiles){
                final /* synthetic */ LongSet val$cleanedFiles;
                {
                    this.val$cleanedFiles = longSet;
                }

                @Override
                public void run() {
                    int filesDeletionDelay = GarbageCollector.this.ec.getGcFilesDeletionDelay();
                    if (filesDeletionDelay == 0) {
                        for (Long file : this.val$cleanedFiles) {
                            GarbageCollector.this.deletionQueue.offer(file);
                        }
                    } else {
                        DeferredIO.getJobProcessor().queueIn(new Job(){

                            protected void execute() throws Throwable {
                                for (Long file : val$cleanedFiles) {
                                    GarbageCollector.this.deletionQueue.offer(file);
                                }
                            }
                        }, (long)filesDeletionDelay);
                    }
                }
            });
        }
        return true;
    }

    private void cleanSingleFile(long fileAddress, @NotNull TransactionImpl txn) {
        if (this.isFileCleaned(fileAddress)) {
            throw new ExodusException("Attempt to clean already cleaned file");
        }
        GarbageCollector.loggingInfo("start cleanFile(" + this.env.getLocation() + File.separatorChar + LogUtil.getLogFilename(fileAddress) + ')');
        Log log = this.getLog();
        if (logger.isDebugEnabled()) {
            long high = log.getHighAddress();
            long highFile = log.getHighFileAddress();
            logger.debug(String.format("Cleaner acquired txn when log high address was: %d (%s@%d) when cleaning file %s", high, LogUtil.getLogFilename(highFile), high - highFile, LogUtil.getLogFilename(fileAddress)));
        }
        try {
            RandomAccessLoggable loggable;
            long nextFileAddress = fileAddress + log.getFileLengthBound();
            LoggableIterator loggables = log.getLoggableIterator(fileAddress);
            while (loggables.hasNext() && (loggable = (RandomAccessLoggable)loggables.next()).getAddress() < nextFileAddress) {
                int structureId = loggable.getStructureId();
                if (structureId == 0 || structureId == 1) continue;
                StoreImpl store = (StoreImpl)this.openStoresCache.get(structureId);
                if (store == null) {
                    store = txn.openStoreByStructureId(structureId);
                    this.openStoresCache.put(structureId, (Object)store);
                }
                store.reclaim(txn, loggable, loggables);
            }
        }
        catch (Throwable e) {
            logger.error("cleanFile(" + LogUtil.getLogFilename(fileAddress) + ')', e);
            throw e;
        }
    }

    private void removeFile(long file) {
        this.getLog().removeFile(file, this.ec.getGcRenameFiles() ? RemoveBlockType.Rename : RemoveBlockType.Delete);
    }
}

