/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

/**
 * This module contains functionality for sending email messages.
 * These messages are independent from the notification scheme that is used for subscriptions to issue updates in YouTrack.
 * All of the components for these email messages are defined in the parameters for the function that is supported in this module.
 * However, the messages are still sent by the SMTP server that is connected to your YouTrack installation.
 * @module @jetbrains/youtrack-scripting-api/notifications
 */

var env = require('./environment');

/**
 * Sends an email message to one or more email addresses.
 * If the project that the issue belongs to uses a dedicated project 'From' email, the email is sent from this address.
 * Otherwise, the default server 'From' address is taken.
 *
 *
 * @example
 * var issue = ctx.issue;
 * if (issue.comments.added.isNotEmpty()) {
 *   var authorName = issue.comments.added.first().author.fullName;
 *   var message = {
 *     fromName: authorName,
 *     toEmails: ['user1@jetbrains.com', 'user2@jetbrains.com'],
 *     subject: 'New comment in ' + issue.id,
 *     body: [
 *       '<div style="font-family: sans-serif">',
 *       '  <div style="padding: 10px 10px; font-size: 13px; border-bottom: 1px solid #D4D5D6;">',
 *       '    New comment was added in issue ' + issue.id + ' by ' + authorName,
 *       '  </div>',
 *       '<\div>'
 *     ].join('\n');
 *   };
 *   notifications.sendEmail(message, issue);
 * }
 *
 * @param {Object} message An object that contains all of the message components that are required for sendout.
 * @param {string} [message.fromName] The visible name of the sender.
 * When used, the specified value is displayed as the sender name in the email message.
 * If the sender email (whether taken from the project settings or the server setting) includes a sender name, it is overwritten with this value.
 * If this parameter is left empty, the presentation of the sender is determined by the 'From' address that is saved in either the project or server settings.
 * @param {string[]} message.toEmails A list of recipient email addresses. The first email address in the array is set as the recipient. All others are set as CC.
 * @param {string} [message.subject] The email message subject.
 * @param {string} [message.body] The email message body.
 * @param {Issue} [issue] The issue that the email message is related to. All email messages that are related to a single issue are combined into one thread.
 */

exports.sendEmail = function (message, issue) {
  if (!message) {
    throw 'Message parameter cannot be undefined';
  }
  var to = message.toEmails;
  if (!to || !to.length) {
    throw 'Message contains an undefined list of recipient email addresses or the list is empty';
  }

  env.sendMailExported(
    message.fromName || '',
    to[0],
    to.length > 1 ? to.slice(1, to.length).join(',') : '',
    message.subject || '',
    message.body || '',
    issue ? (issue.id || '') : '');
};
