/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

var entities = require('@jetbrains/youtrack-scripting-api/entities');
var constants = require('./constants');
var jiraImportContext = require('./importContext');

var toDebugString = function(event) {
  var id = event.changeGroup ? (event.changeGroup.id + '-' + event.indexInGroup) : '0-0';
  return event.jiraIssue.key + ':event:' + id + ':' + event.jiraFieldId;
};

var constructResolutionHistoryItems = function(jiraIssue) {
  var historyItems = [];
  var resolvedTimestamp = null;
  var fieldId = '$resolution_timestamp';
  jiraIssue.importEvents.forEach(function (event) {
    if ('resolution' === event.jiraFieldId) {
      var from = event.change.from ? resolvedTimestamp : null;
      var to = event.change.to ? event.timestamp : null;
      var resolvedTimeChange = {
        field: fieldId,
        fieldtype: 'jira',
        from: from,
        fromString: from,
        to: to,
        toString: to
      };
      attachImportEvents(jiraIssue, fieldId, event.target, event.author, event.timestamp, resolvedTimeChange);
      resolvedTimestamp = to;
    }
  });
  jiraIssue.fields[fieldId] = resolvedTimestamp;
  return historyItems;
};

var createChange = function(from, to, modify) {
  return {
    from: modify ? modify(from ? from : null) : (from ? from : null),
    to: modify ? modify(to ? to : null) : (to ? to : null)
  };
};

exports.extractFromId = function (modify) {
  return function(event) {
    return createChange(event.change.from, event.change.to, modify)
  };
};

exports.extractFromString = function (modify) {
  return function(event) {
    return createChange(event.change.fromString, event.change.toString, modify)
  };
};

exports.splitToArray = function (separator, modify) {
  return function(s) {
    if (!s) {
      return null;
    }
    separator = separator || ',';
    return s.split(separator).filter(function (str) {
      return str && str.length > 0;
    }).map(function (str) {
      return modify ? modify(str.trim()) : str.trim();
    });
  };
};

exports.stripBrackets = function (modify) {
  return function (s) {
    return s ? (modify ? modify(s.slice(1, -1)) : s.slice(1, -1)) : s
  };
};

exports.field = function (event, ytFieldName) {
  entities.ImportedPersistentEvent.createFieldEvent(event.target, event.author, event.timestamp, ytFieldName,
    event.from, event.to);
};

exports.property =  function (event, ytFieldName) {
  entities.ImportedPersistentEvent.createPropEvent(event.target, event.author, event.timestamp, ytFieldName,
    event.from, event.to);
};

exports.addEntity = function (event) {
  entities.ImportedPersistentEvent.createAddEvent(event.target, event.author, event.timestamp);
};

exports.removeEntity = function (event) {
  entities.ImportedPersistentEvent.createRemoveEvent(event.target, event.author, event.timestamp);
};

exports.link = function (event) {
  var linkInfo = event.jiraIssue.key + '->' + event.change.to;
  if (!event.to) {
    console.debug(toDebugString(event) + ':conversion failed, could not find target issue for link ' + linkInfo);
    return;
  }
  var link = event.jiraIssue.importLinks ? event.jiraIssue.importLinks[event.to] : null;
  if (!link) {
    console.debug(toDebugString(event) + ':conversion failed, current version of jira issue has no link ' + linkInfo);
    return;
  }
  entities.ImportedPersistentEvent.createStubLinkEvent(event.target, event.author, event.timestamp, link.typeName, link.outward, link.toId);
};

var attachImportEvents = exports.attachImportEvents = function (jiraIssue, jiraFieldId, ytTarget, ytAuthor, timestamp, jiraChangeItem, jiraChange, indexInGroup) {
  jiraIssue.importEvents.push({
    jiraIssue: jiraIssue,
    jiraFieldId: jiraFieldId,
    target: ytTarget,
    author: ytAuthor,
    timestamp: timestamp,
    change: jiraChangeItem,
    changeGroup: jiraChange,
    indexInGroup: indexInGroup
  });
};

exports.convertEvents = function (events, jiraIssue, issue, findFieldConverter, extract) {
  var jiraProjectKey = jiraIssue.fields.project.key;
  events = events.concat(constructResolutionHistoryItems(jiraIssue));
  events.sort(function(a, b) {return a.timestamp - b.timestamp;});
  var meaningfulEvents = [];
  for (var i = 0; i < events.length; i++) {
    var event = events[i];
    var jiraFieldId = event.jiraFieldId;
    var jiraFieldPrototype = jiraImportContext.fieldSchema.prototypes[jiraFieldId];
    var jiraProject = jiraImportContext.fieldSchema.projects[jiraProjectKey];
    var jiraProjectField = jiraProject.fields[jiraFieldId];
    var converter = findFieldConverter(jiraFieldId, jiraFieldPrototype, jiraProjectField);

    if (extract) {
      var ytFieldName = converter.convertName(jiraFieldPrototype);
      if (converter.convertSchema !== jiraImportContext.no_op) {
        console.debug(toDebugString(event) + ':extraction started');
        var values = [];
        var fromStrings = function (str) {
          return str ? (Array.isArray(str) ? str.map(function (it) {
            return {name: it};
          }) : [{name: str}]) : [];
        };
        var change = converter.transformEvent(event);
        if (change) {
          values = fromStrings(change.from);
          values = values.concat(fromStrings(change.to));
        }
        converter.convertSchema(ytFieldName, issue.project, jiraProject, jiraFieldPrototype, {allowedValues: values});
      } else {
        console.debug(toDebugString(event) + ':extraction skipped');
      }
    } else {
      if (converter.convertEvent !== jiraImportContext.no_op) {
        console.debug(toDebugString(event) + ':conversion started');
        var change = converter.transformEvent(event);
        if (change) {
          event.from = change.from;
          event.to = change.to;
        }
        try {
          converter.convertEvent(event, converter.convertName(jiraFieldPrototype), issue);
        } catch (error) {
          var errorText = toDebugString(event) + ':conversion failed. Check that Youtrack has no corresponding field of different type. ';
          console.error(errorText);
          error.message = errorText + error.message;
          throw error;
        }
        meaningfulEvents.push(event);
      } else {
        console.debug(toDebugString(event) + ':conversion skipped');
      }
    }
  }
  return meaningfulEvents;
};

exports.findFieldId = function(changeFieldId) {
  if (constants.skipEventsOn.indexOf(changeFieldId) > -1) {return null;}
  var jiraFieldId;
  // assume changeFieldId is a field id
  if (jiraImportContext.fieldSchema.prototypes[changeFieldId]) jiraFieldId = changeFieldId;
  // assume changeFieldId is a field name
  if (!jiraFieldId) jiraFieldId = jiraImportContext.fieldSchema.nameToId[changeFieldId];
  // custom mapping
  if (!jiraFieldId) jiraFieldId = constants.weirdChangeFieldIds[changeFieldId];
  return jiraFieldId;
};

exports.findChange = function (event, jiraFieldName) {
  var changeGroup = event.changeGroup;
  var filteredItems = changeGroup.items.filter(function (item) {
    return item.field === jiraFieldName
  });
  return filteredItems.length === 1 ? filteredItems[0] : null;
};
