/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

var importUtils = require('@jetbrains/youtrack-scripting-api-import/utils');
var entities = require('@jetbrains/youtrack-scripting-api/entities');
var events = require('./events');
var importContext = require('./importContext');
var perTypeConverters = require('./perTypeConverters');
var perFieldConverters = require('./perFieldConverters');

var unknownCFsConverter = {
  convertName: function (jiraFieldPrototype) {
    console.warn('Unsupported custom field type ' + jiraFieldPrototype.schema.custom);
    return 'unsupported';
  },
  convertSchema: importContext.no_op,
  convertValue: importContext.no_op,
  convertEvent: importContext.no_op,
  transformEvent: importContext.no_op
};

exports.findFieldConverter = function (jiraFieldId, jiraFieldPrototype, jiraProjectField) {
  var perTypeConverter;
  var jiraFieldType;
  var custom;
  if (jiraFieldPrototype && jiraFieldPrototype.schema) {
    custom = jiraFieldPrototype.schema.custom;
    if (jiraFieldPrototype.custom && custom &&
      (custom.indexOf('com.atlassian.jira.plugin.system.customfieldtypes:') === -1 ||
        custom === 'com.atlassian.jira.plugin.system.customfieldtypes:cascadingselect')) {
      jiraFieldType = custom
    } else if (jiraFieldPrototype.schema.type === 'array') {
      if (custom === 'com.atlassian.jira.plugin.system.customfieldtypes:labels') {
        jiraFieldType = 'label[*]';
      } else {
        jiraFieldType = jiraFieldPrototype.schema.items + '[*]';
      }
    } else if (jiraFieldPrototype.schema.system === 'assignee') {
      jiraFieldType = 'user[1]';
    } else if (jiraProjectField && jiraProjectField.allowedValues && Array.isArray(jiraProjectField.allowedValues) ||
      custom === 'com.atlassian.jira.plugin.system.customfieldtypes:select' ||
      custom === 'com.atlassian.jira.plugin.system.customfieldtypes:radiobuttons') {
      jiraFieldType = jiraFieldPrototype.schema.type + '[1]';
    } else {
      jiraFieldType = jiraFieldPrototype.schema.type;
    }
    perTypeConverter = perTypeConverters[jiraFieldType];
  }

  var schema = function (converter) {
    var con = '';
    con += converter.convertName ? '1' : '0';
    con += converter.convertSchema ? '1' : '0';
    con += converter.convertValue ? '1' : '0';
    con += converter.convertEvent ? '1' : '0';
    con += converter.transformEvent ? '1' : '0';
    return con;
  };

  var perFieldConverter = perFieldConverters[jiraFieldId];
  if (!perFieldConverter && !perTypeConverter) {
    console.error('Unsupported custom field {jira: ' + jiraFieldId + '<' + jiraFieldType + '>}');
    return unknownCFsConverter;
  }

  var fSchema = perFieldConverter ? schema(perFieldConverter) : '-----';
  var tSchema = perTypeConverter ? schema(perTypeConverter) : '-----';
  console.trace('Converters for field {jira: ' + jiraFieldId + '<' + jiraFieldType + '>}: ' + fSchema + '<' + tSchema + '>');

  perFieldConverter = perFieldConverter || {};
  perTypeConverter = perTypeConverter || {};

  var converter = {
    convertSchema: perFieldConverter.convertSchema || perTypeConverter.convertSchema || importContext.no_op,
    convertValue: perFieldConverter.convertValue || perTypeConverter.convertValue || importContext.no_op,
    convertEvent: perFieldConverter.convertEvent || perTypeConverter.convertEvent || importContext.no_op,
    transformEvent: perFieldConverter.transformEvent || perTypeConverter.transformEvent || importContext.no_op
  };

  var customConvertName = perFieldConverter.convertName || perTypeConverter.convertName;
  var defaultConvertName = perFieldConverters.renamer(jiraFieldPrototype ? jiraFieldPrototype.name :
    (jiraProjectField ? jiraProjectField.name : null));

  converter.convertName = function (jiraFieldPrototype) {
    var unescapedName = (customConvertName || defaultConvertName)(jiraFieldPrototype);
    var requestedName = importUtils.unforbid(unescapedName);
    if (jiraFieldPrototype) {
      var ytPrototype;
      if (custom || (perFieldConverter && !perFieldConverter.embedded)) {
        var requestedType = (perFieldConverter && perFieldConverter.convertType) || perTypeConverter.convertType;
        if (!requestedType) {
          return requestedName
        }
        var dominantFieldId = perFieldConverter && perFieldConverter.dominantFieldId;
        ytPrototype = findOrCreatePrototype(requestedName, requestedType, jiraFieldType, jiraFieldPrototype, dominantFieldId);
      } else {
        ytPrototype = {
          name: requestedName,
          getTypeName: function () {
            return 'undefined';
          }
        };
      }
      jiraFieldPrototype.$ytPrototype = ytPrototype;
      jiraFieldPrototype.$typeName = jiraFieldType;
      return ytPrototype.name;
    } else {
      // project custom fields with no prototype (e.g. parent)
      return requestedName;
    }
  };

  return converter;
};

var findOrCreatePrototype = function (ytRequestedName, ytRequestedType, jiraType, jiraFieldPrototype, dominantFieldId) {
  var jiraFieldInfo = '{jira: ' + jiraFieldPrototype.id + '<' + jiraType + '>}';
  var ytFieldPrototype = jiraFieldPrototype.$ytPrototype;
  ytFieldPrototype && console.trace('Found field association: ' + association(jiraFieldInfo, ytFieldPrototype) + ' in cache');

  var jiraId = dominantFieldId || jiraFieldPrototype.id;
  if (!ytFieldPrototype) {
    ytFieldPrototype = entities.CustomFieldPrototype.findByImportId(jiraId);
    ytFieldPrototype && console.trace('Found field association: ' + association(jiraFieldInfo, ytFieldPrototype) + ' by jira id ' + jiraId);
  }

  if (!ytFieldPrototype) {
    ytFieldPrototype = findOrCreatePrototypeByName(ytRequestedName, ytRequestedType, jiraFieldInfo, jiraFieldPrototype, jiraId);
    ytFieldPrototype && console.trace('Created new field association: ' + association(jiraFieldInfo, ytFieldPrototype));
  }

  if (ytRequestedType !== ytFieldPrototype.typeName) {
    importUtils.throw('Invalid association: ' + association(jiraFieldInfo, ytFieldPrototype) + ', fields have different types');
  }
  if (!ytFieldPrototype.jiraId) {
    importUtils.throw('Invalid association: ' + association(jiraFieldInfo, ytFieldPrototype) + ', YT prototype has no jira id');
  }
  return ytFieldPrototype;
};

var findOrCreatePrototypeByName = function (ytName, ytType, jiraFieldInfo, jiraFieldPrototype, jiraId) {
  var ytFieldPrototype = entities.CustomFieldPrototype.findByNameOrLocalizedName(ytName);
  if (!ytFieldPrototype) {
    ytName = findUniqueNameForCustomFieldPrototype(ytName, predefinedFieldNames);
    var old = entities.CustomFieldPrototype.findByNameOrLocalizedName(ytName);
    if (!old) {
      ytFieldPrototype = entities.CustomFieldPrototype.create(ytName, ytType);
    } else {
      ytFieldPrototype = findOrCreatePrototypeByName(ytName, ytType, jiraFieldInfo, jiraFieldPrototype, jiraId);
    }
    ytFieldPrototype.jiraId = jiraId;
    console.info('Associated new YT prototype: ' + association(jiraFieldInfo, ytFieldPrototype));
  } else {
    var failed = null;
    if (ytFieldPrototype.jiraId) {
      failed = 'YT prototype associated with another jira field: ' + ytFieldPrototype.jiraId;
    } else if (ytFieldPrototype.typeName !== ytType) {
      failed = 'YT prototype has inappropriate type: ' + ytFieldPrototype.typeName;
    }
    if (failed) {
      console.warn('Failed to create association: ' + association(jiraFieldInfo, ytFieldPrototype) + ', ' + failed);
      var ytUglyName = importUtils.uglify(ytName);
      ytFieldPrototype = findOrCreatePrototypeByName(ytUglyName, ytType, jiraFieldInfo, jiraFieldPrototype, jiraId);
    } else {
      ytFieldPrototype.jiraId = jiraId;
      console.info('Associated existed YT prototype: ' + association(jiraFieldInfo, ytFieldPrototype));
    }
  }
  return ytFieldPrototype;
};

var findUniqueNameForCustomFieldPrototype = function (name, predefinedFieldNames) {
    if (predefinedFieldNames.indexOf(name.toLowerCase()) !== -1) {
        return findUniqueNameForCustomFieldPrototype(importUtils.uglify(name), predefinedFieldNames);
    } else {
        return name;
    }
};

var association = function(jiraFieldInfo, ytFieldPrototype) {
  return jiraFieldInfo + ' ~ {yt: ' + ytFieldPrototype.name + '<' + ytFieldPrototype.typeName + '>[' + ytFieldPrototype.jiraId + ']}';
};
