/*
 Copyright 2017 JetBrains s.r.o.

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

var jira = require('./jira');
var cache = require('@jetbrains/youtrack-scripting-api/cache');
var importUtils = require('@jetbrains/youtrack-scripting-api-import/utils');
var transaction = require('@jetbrains/youtrack-scripting-api-import/transaction');
var params = require('./importSettings');

var getOrCreate = function(name, constructor) {
  var value = ctx.get().getObject(name);
  if (!value) {
    value = constructor();
    ctx.get().setObject(name, value);
  }
  return value;
};

var thingsToReexportFromContext = {
  timeInterval: function () {
    return {
      from: {},
      to: '',
      includes: function (projectKey, date) {
        var from = this.from[projectKey];
        return date < this.to && (!from || date > from )
      }
    };
  },
  progress: function () {
    return {
      issuesAdded: 0,
      issuesUpdated: 0
    };
  },
  jiraClient: function () {
    return new jira.JiraClient(params.url, params.sslKeyName, params.login, params.password);
  },
  linkTypes: function () {
    return {
      byJiraId: {},
      byTypeName: {},
      byLinkName: {}
    }
  },
  fieldSchema: function () {
    var thisRef = {
      prototypes: {},
      projects: {},
      nameToId: {},
      statuses: null,
      resolutions: null,
      visitFieldPrototype: function (jiraFieldPrototype) {
        thisRef.prototypes[jiraFieldPrototype.id] = jiraFieldPrototype;
        thisRef.nameToId[jiraFieldPrototype.name] = jiraFieldPrototype.id;
      },
      visitProject: function (jiraProject) {
        var jiraProjectKey = jiraProject.key;
        thisRef.statuses = this.statuses || exports.jiraClient.getStatuses(networkFailureHandler);
        thisRef.resolutions = this.resolutions || exports.jiraClient.getResolutions(networkFailureHandler);

        jiraProject.$permissions = exports.jiraClient.getProjectPermissions(jiraProjectKey, networkFailureHandler);
        console.info(jiraProjectKey + ' permissions: ' + Object.keys(jiraProject.$permissions).map(function (key) {
            return key + ':' + jiraProject.$permissions[key]
          }).join());

        if (!jiraProject.$permissions['BROWSE']) {
          console.log("Skip project " + jiraProjectKey + " as you do not have BROWSE permission for it.");
          return;
        }
        var jiraProjectFields = {};
        if (jiraProject.$permissions['CREATE_ISSUE']) {
          var processIssueType = function (issueType) {
            var fields = issueType.fields;
            Object.keys(fields).forEach(function (jiraFieldId) {
              var field = fields[jiraFieldId];
              if (jiraFieldId === 'issuetype' && jiraProjectFields[jiraFieldId]) {
                jiraProjectFields[jiraFieldId].allowedValues.push(field.allowedValues[0]);
              } else {
                jiraProjectFields[jiraFieldId] = field;
              }
            });
          };
          exports.jiraClient.getIssueCreateMeta(jiraProjectKey, networkFailureHandler).projects[0].issuetypes.forEach(processIssueType);
        } else {
          console.info('Can not retrieve IssueCreateMeta for ' + jiraProjectKey + ' project, no CREATE_ISSUE permission');
        }

        // if there is no jira project field 'assignee' create this field artificially if we have permissions to get the assignees
        var jiraAssignees = [];
        /*
         if (!jiraProjectFields['assignee'] && (jiraProject.$permissions['PROJECT_ADMIN'] || jiraProject.$permissions['ASSIGN_ISSUE'])) {
         exports.jiraClient.getAssignees(jiraProject.key, function (jiraUserPack) {
         jiraAssignees.push.apply(jiraAssignees, jiraUserPack);
         return true;
         }, networkFailureHandler);
         }
         */
        jiraProjectFields['assignee'] = {
          required: false,
          schema: {
            type: 'user',
            system: 'assignee'
          },
          allowedValues: jiraAssignees
        };

        jiraProject.fields = jiraProjectFields;
        thisRef.projects[jiraProjectKey] = jiraProject;
        console.info('Loaded project info for ' + jiraProjectKey);
      }
    };
    return thisRef
  },
  hubSyncVariables: function () {
    return {
      preLastUserExportDuration: 10000,   // just an initial value, it took like ages
      lastUserExportStart: now() - 10000, // the same here
      lastUserExportEnd: now(),           // just an initial value
      adaptiveIncrease: 0
    }
  },
  jsonsCache: function () {
    return cache.create(100, 'JSONs');
  },
  usersCache: function () {
    return cache.create(1000, 'users');
  }
};

// publish
Object.keys(thingsToReexportFromContext).forEach(function (key) {
  Object.defineProperty(exports, key, {
      enumerable: true,
      get: function () {
        return getOrCreate(key, thingsToReexportFromContext[key])
      }
    }
  )
});

Object.defineProperty(exports, 'userTimeZone', {
    enumerable: true,
    get: function () {
      return ctx.get().getObject('userTimeZone');
    },
    set: function (val) {
      ctx.get().setObject('userTimeZone', val)
    }
  }
);

var networkFailureHandler = exports.networkFailureHandler = function(failure) {
  transaction.revert();
  if (failure.response) {
    var errorJson = null;
    try {
      errorJson = JSON.parse(failure.response);
    } catch (e) {
      // ignore, html can be here, not json
    }
    importUtils.throw(errorJson ? (failure.code + ' ' + errorJson.errorMessages) :
          ('Jira replied with status code ' + failure.code));
  } else if (failure.exception) {
    importUtils.throw(failure.exception);
  }
};

// TODO extract code common to this function and the one above
exports.attachmentContentFailureHandler = function(failure) {
  if (failure.response) {
    var errorJson = null;
    try {
      errorJson = JSON.parse(failure.response);
    } catch (e) {
      // ignore, html can be here, not json
    }
    console.error(errorJson ? (failure.code + ' ' + errorJson.errorMessages) :
      ('Jira replied with status code ' + failure.code + ' during attachment content request'));
  } else if (failure.exception) {
    console.error(failure.exception)
  }
};

exports.getSkippingFailureHandler = function(statusCodesToSkip) {
  return function (failure) {
    failure.code && (statusCodesToSkip.indexOf(failure.code) !== -1) || networkFailureHandler(failure);
  }
};

exports.hasPermission = function (permission, projectKey) {
  return exports.fieldSchema.projects[projectKey].$permissions[permission];
};

exports.no_op = function () {};